<?php
require "conn.php";

// Handle Delete
if (isset($_GET['delete'])) {
    $mill_id = intval($_GET['delete']);
    $delete_query = "DELETE FROM mills WHERE mill_id = ?";
    $stmt = mysqli_prepare($conn, $delete_query);
    mysqli_stmt_bind_param($stmt, "i", $mill_id);
    
    if (mysqli_stmt_execute($stmt)) {
        $success_msg = "Mill deleted successfully!";
    } else {
        $error_msg = "Error deleting mill: " . mysqli_error($conn);
    }
    mysqli_stmt_close($stmt);
}

// Handle Insert/Update
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $mill_id = isset($_POST['mill_id']) ? intval($_POST['mill_id']) : 0;
    $mill_name = trim($_POST['mill_name']);
    $permit_type = trim($_POST['permit_type']);
    $license_number = trim($_POST['license_number']);
    $total_authorized_qty = intval($_POST['total_authorized_qty']);
    $unit = trim($_POST['unit']);
    $street_address = trim($_POST['street_address']);
    $dispatch_province = trim($_POST['dispatch_province']);
    $dispatch_division = trim($_POST['dispatch_division']);
    $dispatch_district = trim($_POST['dispatch_district']);
    $status = $_POST['status'];
    
    // Validation
    if (empty($mill_name) || empty($permit_type) || empty($license_number) || empty($street_address) || empty($dispatch_province) || empty($dispatch_division) || empty($dispatch_district) || $total_authorized_qty <= 0) {
        $error_msg = "All fields are required and quantity must be greater than 0!";
    } else {
        if ($mill_id > 0) {
            // Update
            $update_query = "UPDATE mills SET mill_name = ?, permit_type = ?, license_number = ?, total_authorized_qty = ?, unit = ?, street_address = ?, dispatch_province = ?, dispatch_division = ?, dispatch_district = ?, status = ? WHERE mill_id = ?";
            $stmt = mysqli_prepare($conn, $update_query);
            mysqli_stmt_bind_param($stmt, "ssisssssssi", $mill_name, $permit_type, $license_number, $total_authorized_qty, $unit, $street_address, $dispatch_province, $dispatch_division, $dispatch_district, $status, $mill_id);
            
            if (mysqli_stmt_execute($stmt)) {
                $success_msg = "Mill updated successfully!";
            } else {
                $error_msg = "Error updating mill: " . mysqli_error($conn);
            }
        } else {
            // Insert
            $insert_query = "INSERT INTO mills (mill_name, permit_type, license_number, total_authorized_qty, unit, street_address, dispatch_province, dispatch_division, dispatch_district, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt = mysqli_prepare($conn, $insert_query);
            mysqli_stmt_bind_param($stmt, "ssisssssss", $mill_name, $permit_type, $license_number, $total_authorized_qty, $unit, $street_address, $dispatch_province, $dispatch_division, $dispatch_district, $status);
            
            if (mysqli_stmt_execute($stmt)) {
                $success_msg = "Mill added successfully!";
            } else {
                $error_msg = "Error adding mill: " . mysqli_error($conn);
            }
        }
        mysqli_stmt_close($stmt);
    }
}

// Fetch mill for editing
$edit_mill = null;
if (isset($_GET['edit'])) {
    $edit_id = intval($_GET['edit']);
    $edit_query = "SELECT * FROM mills WHERE mill_id = ?";
    $stmt = mysqli_prepare($conn, $edit_query);
    mysqli_stmt_bind_param($stmt, "i", $edit_id);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    $edit_mill = mysqli_fetch_assoc($result);
    mysqli_stmt_close($stmt);
}

// Fetch all mills
$mills_query = "SELECT * FROM mills ORDER BY mill_id DESC";
$mills_result = mysqli_query($conn, $mills_query);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <title>FDIMS :: Mills Management</title>
    <link rel="shortcut icon" href="assets/images/favicon.png">
    <link href="https://fonts.googleapis.com/css?family=Roboto:400,500,700&display=swap" rel="stylesheet">
    <link href="font-awesome/4.7.0/css/font-awesome.min.css" rel="stylesheet">
    <link href="assets/css/bootstrap.min.css" rel="stylesheet" type="text/css">
    <link href="assets/css/style.css" rel="stylesheet" type="text/css">
    <link href="assets/css/custom.css" rel="stylesheet" type="text/css">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 20px;
            min-height: 100vh;
        }
        .container-custom {
            max-width: 1400px;
            margin: 0 auto;
            background: #fff;
            padding: 35px;
            border-radius: 12px;
            box-shadow: 0 8px 30px rgba(0,0,0,0.15);
        }
        .page-header {
            border-bottom: 3px solid #007bff;
            padding-bottom: 20px;
            margin-bottom: 35px;
        }
        .page-header h2 {
            color: #2c3e50;
            font-weight: 600;
            margin-bottom: 5px;
        }
        .alert {
            padding: 15px 20px;
            margin-bottom: 25px;
            border-radius: 6px;
            border-left: 4px solid;
        }
        .alert-success {
            background-color: #d4edda;
            border-left-color: #28a745;
            color: #155724;
        }
        .alert-danger {
            background-color: #f8d7da;
            border-left-color: #dc3545;
            color: #721c24;
        }
        .form-section {
            background: linear-gradient(to bottom, #f8f9fa 0%, #e9ecef 100%);
            padding: 30px;
            border-radius: 8px;
            margin-bottom: 40px;
            border: 1px solid #dee2e6;
        }
        .form-section h4 {
            color: #495057;
            font-weight: 600;
            margin-bottom: 25px;
            padding-bottom: 10px;
            border-bottom: 2px solid #007bff;
        }
        .table-section {
            margin-top: 40px;
        }
        .table-section h4 {
            color: #495057;
            font-weight: 600;
            margin-bottom: 20px;
        }
        .table-responsive {
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        }
        .table {
            margin-bottom: 0;
            font-size: 14px;
        }
        .table thead th {
            background: linear-gradient(135deg, #007bff 0%, #0056b3 100%);
            color: white;
            border: none;
            padding: 15px 12px;
            font-weight: 600;
            text-transform: uppercase;
            font-size: 12px;
            letter-spacing: 0.5px;
            vertical-align: middle;
            text-align: center;
        }
        .table tbody td {
            padding: 12px;
            vertical-align: middle;
            border-color: #e9ecef;
            text-align: center;
        }
        .table tbody tr {
            transition: all 0.3s ease;
        }
        .table tbody tr:hover {
            background-color: #f1f3f5;
            transform: scale(1.01);
            box-shadow: 0 2px 5px rgba(0,0,0,0.05);
        }
        .btn-action {
            padding: 6px 14px;
            font-size: 12px;
            font-weight: 500;
            border-radius: 4px;
            transition: all 0.3s ease;
            border: none;
            min-width: 70px;
            display: inline-block;
            text-align: center;
        }
        .btn-action i {
            margin-right: 4px;
        }
        .btn-action:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0,0,0,0.2);
        }
        .btn-sm.btn-info {
            background-color: #17a2b8;
            color: white;
        }
        .btn-sm.btn-info:hover {
            background-color: #138496;
        }
        .btn-sm.btn-danger {
            background-color: #dc3545;
            color: white;
        }
        .btn-sm.btn-danger:hover {
            background-color: #c82333;
        }
        .badge {
            padding: 6px 12px;
            border-radius: 4px;
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        .badge-success {
            background-color: #28a745;
            color: white;
        }
        .badge-secondary {
            background-color: #6c757d;
            color: white;
        }
        .btn-primary {
            background: linear-gradient(135deg, #007bff 0%, #0056b3 100%);
            border: none;
            padding: 10px 25px;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0,123,255,0.3);
        }
        .btn-secondary {
            background-color: #6c757d;
            border: none;
            padding: 10px 25px;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        .btn-secondary:hover {
            background-color: #5a6268;
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(108,117,125,0.3);
        }
        .form-control {
            border-radius: 4px;
            border: 1px solid #ced4da;
            padding: 8px 12px;
            transition: all 0.3s ease;
        }
        .form-control:focus {
            border-color: #007bff;
            box-shadow: 0 0 0 0.2rem rgba(0,123,255,0.25);
        }
        .action-buttons {
            white-space: nowrap;
        }
    </style>
</head>
<body>
    <div class="container-custom">
        <div class="page-header">
            <h2><i class="fa fa-industry"></i> Mills Management</h2>
            <p class="text-muted">Add, Edit, and Delete Flour Mills</p>
        </div>

        <?php if (isset($success_msg)): ?>
            <div class="alert alert-success">
                <i class="fa fa-check-circle"></i> <?php echo htmlspecialchars($success_msg); ?>
            </div>
        <?php endif; ?>

        <?php if (isset($error_msg)): ?>
            <div class="alert alert-danger">
                <i class="fa fa-exclamation-circle"></i> <?php echo htmlspecialchars($error_msg); ?>
            </div>
        <?php endif; ?>

        <!-- Form Section -->
        <div class="form-section">
            <h4><?php echo $edit_mill ? 'Edit Mill' : 'Add New Mill'; ?></h4>
            <form method="POST" action="mills_manage.php">
                <?php if ($edit_mill): ?>
                    <input type="hidden" name="mill_id" value="<?php echo $edit_mill['mill_id']; ?>">
                <?php endif; ?>
                
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label>Mill Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="mill_name" 
                                   value="<?php echo $edit_mill ? htmlspecialchars($edit_mill['mill_name']) : ''; ?>" 
                                   required>
                        </div>
                    </div>
                    
                    <div class="col-md-6">
                        <div class="form-group">
                            <label>Permit Type <span class="text-danger">*</span></label>
                            <select class="form-control" name="permit_type" required>
                                <option value="">--Select Permit Type--</option>
                                <option value="Wheat" <?php echo ($edit_mill && $edit_mill['permit_type'] == 'Wheat') ? 'selected' : ''; ?>>Wheat</option>
                                <option value="Rice" <?php echo ($edit_mill && $edit_mill['permit_type'] == 'Rice') ? 'selected' : ''; ?>>Rice</option>
                                <option value="Corn" <?php echo ($edit_mill && $edit_mill['permit_type'] == 'Corn') ? 'selected' : ''; ?>>Corn</option>
                                <option value="Mixed" <?php echo ($edit_mill && $edit_mill['permit_type'] == 'Mixed') ? 'selected' : ''; ?>>Mixed</option>
                            </select>
                        </div>
                    </div>
                </div>

                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label>License Number <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="license_number" 
                                   value="<?php echo $edit_mill ? htmlspecialchars($edit_mill['license_number']) : ''; ?>" 
                                   required>
                        </div>
                    </div>
                    
                    <div class="col-md-3">
                        <div class="form-group">
                            <label>Total Authorized Qty <span class="text-danger">*</span></label>
                            <input type="number" class="form-control" name="total_authorized_qty" 
                                   value="<?php echo $edit_mill ? $edit_mill['total_authorized_qty'] : ''; ?>" 
                                   min="1" required>
                        </div>
                    </div>
                    
                    <div class="col-md-3">
                        <div class="form-group">
                            <label>Unit <span class="text-danger">*</span></label>
                            <select class="form-control" name="unit" required>
                                <option value="Tons" <?php echo ($edit_mill && $edit_mill['unit'] == 'Tons') ? 'selected' : ''; ?>>Tons</option>
                                <option value="KG" <?php echo ($edit_mill && $edit_mill['unit'] == 'KG') ? 'selected' : ''; ?>>KG</option>
                                <option value="Bags" <?php echo ($edit_mill && $edit_mill['unit'] == 'Bags') ? 'selected' : ''; ?>>Bags</option>
                            </select>
                        </div>
                    </div>
                </div>

                <div class="row">
                    <div class="col-md-12">
                        <div class="form-group">
                            <label>Street Address <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="street_address" 
                                   value="<?php echo $edit_mill ? htmlspecialchars($edit_mill['street_address']) : ''; ?>" 
                                   required>
                        </div>
                    </div>
                </div>

                <div class="row">
                    <div class="col-md-4">
                        <div class="form-group">
                            <label>Dispatch Province <span class="text-danger">*</span></label>
                            <select class="form-control" name="dispatch_province" required>
                                <option value="">--Select Province--</option>
                                <option value="Punjab" <?php echo ($edit_mill && $edit_mill['dispatch_province'] == 'Punjab') ? 'selected' : ''; ?>>Punjab</option>
                                <option value="Khyber Pakhtunkhwa" <?php echo ($edit_mill && $edit_mill['dispatch_province'] == 'Khyber Pakhtunkhwa') ? 'selected' : ''; ?>>Khyber Pakhtunkhwa</option>
                                <option value="Sindh" <?php echo ($edit_mill && $edit_mill['dispatch_province'] == 'Sindh') ? 'selected' : ''; ?>>Sindh</option>
                                <option value="Balochistan" <?php echo ($edit_mill && $edit_mill['dispatch_province'] == 'Balochistan') ? 'selected' : ''; ?>>Balochistan</option>
                                <option value="Islamabad" <?php echo ($edit_mill && $edit_mill['dispatch_province'] == 'Islamabad') ? 'selected' : ''; ?>>Islamabad</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="form-group">
                            <label>Dispatch Division <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="dispatch_division" 
                                   value="<?php echo $edit_mill ? htmlspecialchars($edit_mill['dispatch_division']) : ''; ?>" 
                                   required>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="form-group">
                            <label>Dispatch District <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="dispatch_district" 
                                   value="<?php echo $edit_mill ? htmlspecialchars($edit_mill['dispatch_district']) : ''; ?>" 
                                   required>
                        </div>
                    </div>
                </div>

                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label>Status <span class="text-danger">*</span></label>
                            <select class="form-control" name="status" required>
                                <option value="Active" <?php echo ($edit_mill && $edit_mill['status'] == 'Active') ? 'selected' : ''; ?>>Active</option>
                                <option value="Inactive" <?php echo ($edit_mill && $edit_mill['status'] == 'Inactive') ? 'selected' : ''; ?>>Inactive</option>
                            </select>
                        </div>
                    </div>
                </div>

                <div class="form-group">
                    <button type="submit" class="btn btn-primary">
                        <i class="fa fa-save"></i> <?php echo $edit_mill ? 'Update Mill' : 'Add Mill'; ?>
                    </button>
                    <?php if ($edit_mill): ?>
                        <a href="mills_manage.php" class="btn btn-secondary">
                            <i class="fa fa-times"></i> Cancel
                        </a>
                    <?php endif; ?>
                </div>
            </form>
        </div>

        <!-- Table Section -->
        <div class="table-section">
            <h4>All Mills</h4>
            <div class="table-responsive">
                <table class="table table-bordered table-hover">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Mill Name</th>
                            <th>Permit Type</th>
                            <th>License Number</th>
                            <th>Authorized Qty</th>
                            <th>Address</th>
                            <th>Status</th>
                            <th>Created</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (mysqli_num_rows($mills_result) > 0): ?>
                            <?php while ($mill = mysqli_fetch_assoc($mills_result)): ?>
                                <tr>
                                    <td><?php echo $mill['mill_id']; ?></td>
                                    <td><?php echo htmlspecialchars($mill['mill_name']); ?></td>
                                    <td><?php echo htmlspecialchars($mill['permit_type']); ?></td>
                                    <td><?php echo htmlspecialchars($mill['license_number']); ?></td>
                                    <td><?php echo number_format($mill['total_authorized_qty']) . ' ' . htmlspecialchars($mill['unit']); ?></td>
                                    <td><?php echo htmlspecialchars($mill['street_address']); ?></td>
                                    <td>
                                        <span class="badge badge-<?php echo $mill['status'] == 'Active' ? 'success' : 'secondary'; ?>">
                                            <?php echo $mill['status']; ?>
                                        </span>
                                    </td>
                                    <td><?php echo date('d-m-Y', strtotime($mill['created_at'])); ?></td>
                                    <td class="action-buttons">
                                        <a href="mills_manage.php?edit=<?php echo $mill['mill_id']; ?>" 
                                           class="btn btn-sm btn-info btn-action"
                                           title="Edit Mill">
                                            <i class="fa fa-edit"></i> Edit
                                        </a>
                                        <a href="mills_manage.php?delete=<?php echo $mill['mill_id']; ?>" 
                                           class="btn btn-sm btn-danger btn-action"
                                           title="Delete Mill"
                                           onclick="return confirm('Are you sure you want to delete this mill?');">
                                            <i class="fa fa-trash"></i> Delete
                                        </a>
                                    </td>
                                </tr>
                            <?php endwhile; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="9" class="text-center">No mills found. Add your first mill above.</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <script src="assets/js/jquery.min.js"></script>
    <script src="assets/js/bootstrap.bundle.min.js"></script>
</body>
</html>
